# /// script
# requires-python = ">=3.10"
# dependencies = ["numpy"]
# ///
"""
20 Balls Bouncing Inside a Spinning Heptagon
Physics simulation with gravity, friction, and realistic collisions.
"""

import tkinter as tk
import math
import numpy as np
from dataclasses import dataclass, field
from typing import List, Tuple
import sys

# Constants
WIDTH = 900
HEIGHT = 900
CENTER_X = WIDTH // 2
CENTER_Y = HEIGHT // 2
NUM_BALLS = 20
HEPTAGON_SIDES = 7
HEPTAGON_RADIUS = 350  # Large enough to contain all balls
BALL_RADIUS = 28
GRAVITY = 800  # pixels/s^2
FRICTION = 0.998  # velocity damping per frame
WALL_FRICTION = 0.85  # tangential velocity retention on wall hit
RESTITUTION = 0.75  # bounce coefficient (controls bounce height)
ANGULAR_FRICTION = 0.995  # rotational damping
SPIN_RATE = 2 * math.pi / 5  # 360 degrees per 5 seconds
DT = 1 / 60  # time step (60 FPS target)

# Ball colors
COLORS = [
    "#f8b862", "#f6ad49", "#f39800", "#f08300", "#ec6d51",
    "#ee7948", "#ed6d3d", "#ec6800", "#ec6800", "#ee7800",
    "#eb6238", "#ea5506", "#ea5506", "#eb6101", "#e49e61",
    "#e45e32", "#e17b34", "#dd7a56", "#db8449", "#d66a35"
]


@dataclass
class Ball:
    """Represents a bouncing ball with physics properties."""
    x: float
    y: float
    vx: float = 0.0
    vy: float = 0.0
    radius: float = BALL_RADIUS
    mass: float = 1.0
    angle: float = 0.0  # rotation angle for number display
    angular_velocity: float = 0.0
    color: str = "#ff0000"
    number: int = 1
    
    def update(self, dt: float):
        """Update ball position and rotation."""
        # Apply gravity
        self.vy += GRAVITY * dt
        
        # Apply friction
        self.vx *= FRICTION
        self.vy *= FRICTION
        
        # Update position
        self.x += self.vx * dt
        self.y += self.vy * dt
        
        # Update rotation based on angular velocity
        self.angle += self.angular_velocity * dt
        self.angular_velocity *= ANGULAR_FRICTION


class Heptagon:
    """Represents the rotating heptagon container."""
    
    def __init__(self, cx: float, cy: float, radius: float, sides: int = 7):
        self.cx = cx
        self.cy = cy
        self.radius = radius
        self.sides = sides
        self.angle = 0.0  # current rotation angle
        
    def get_vertices(self) -> List[Tuple[float, float]]:
        """Get current vertex positions considering rotation."""
        vertices = []
        for i in range(self.sides):
            theta = self.angle + (2 * math.pi * i / self.sides) - math.pi / 2
            x = self.cx + self.radius * math.cos(theta)
            y = self.cy + self.radius * math.sin(theta)
            vertices.append((x, y))
        return vertices
    
    def get_edges(self) -> List[Tuple[Tuple[float, float], Tuple[float, float]]]:
        """Get edges as pairs of vertices."""
        vertices = self.get_vertices()
        edges = []
        for i in range(self.sides):
            p1 = vertices[i]
            p2 = vertices[(i + 1) % self.sides]
            edges.append((p1, p2))
        return edges
    
    def update(self, dt: float):
        """Update heptagon rotation."""
        self.angle += SPIN_RATE * dt


def point_to_line_distance(px: float, py: float, x1: float, y1: float, 
                           x2: float, y2: float) -> Tuple[float, float, float, float]:
    """
    Calculate shortest distance from point to line segment.
    Returns: (distance, closest_x, closest_y, t_param)
    """
    dx = x2 - x1
    dy = y2 - y1
    length_sq = dx * dx + dy * dy
    
    if length_sq == 0:
        return math.hypot(px - x1, py - y1), x1, y1, 0
    
    t = max(0, min(1, ((px - x1) * dx + (py - y1) * dy) / length_sq))
    closest_x = x1 + t * dx
    closest_y = y1 + t * dy
    distance = math.hypot(px - closest_x, py - closest_y)
    
    return distance, closest_x, closest_y, t


def get_wall_velocity(heptagon: Heptagon, point_x: float, point_y: float) -> Tuple[float, float]:
    """Calculate the velocity of a point on the rotating wall."""
    # Relative position from center
    rx = point_x - heptagon.cx
    ry = point_y - heptagon.cy
    
    # Tangential velocity due to rotation (v = omega × r)
    vx = -SPIN_RATE * ry
    vy = SPIN_RATE * rx
    
    return vx, vy


def collide_ball_wall(ball: Ball, heptagon: Heptagon):
    """Handle collision between ball and heptagon walls."""
    edges = heptagon.get_edges()
    
    for (x1, y1), (x2, y2) in edges:
        dist, cx, cy, t = point_to_line_distance(ball.x, ball.y, x1, y1, x2, y2)
        
        if dist < ball.radius and dist > 0:
            # Calculate normal (pointing inward)
            nx = ball.x - cx
            ny = ball.y - cy
            n_len = math.hypot(nx, ny)
            if n_len == 0:
                continue
            nx /= n_len
            ny /= n_len
            
            # Move ball out of wall
            overlap = ball.radius - dist
            ball.x += nx * overlap
            ball.y += ny * overlap
            
            # Get wall velocity at collision point
            wall_vx, wall_vy = get_wall_velocity(heptagon, cx, cy)
            
            # Relative velocity
            rel_vx = ball.vx - wall_vx
            rel_vy = ball.vy - wall_vy
            
            # Normal and tangential components
            vn = rel_vx * nx + rel_vy * ny
            
            if vn < 0:  # Only if moving toward wall
                # Tangent vector
                tx = -ny
                ty = nx
                vt = rel_vx * tx + rel_vy * ty
                
                # Apply restitution to normal component
                new_vn = -vn * RESTITUTION
                
                # Apply friction to tangential component
                new_vt = vt * WALL_FRICTION
                
                # Reconstruct velocity in world frame
                ball.vx = wall_vx + new_vn * nx + new_vt * tx
                ball.vy = wall_vy + new_vn * ny + new_vt * ty
                
                # Update angular velocity based on tangential friction
                # Rolling effect: angular velocity relates to tangential velocity
                rolling_factor = 0.3
                ball.angular_velocity += (vt - new_vt) * rolling_factor / ball.radius


def collide_balls(ball1: Ball, ball2: Ball):
    """Handle collision between two balls."""
    dx = ball2.x - ball1.x
    dy = ball2.y - ball1.y
    dist = math.hypot(dx, dy)
    min_dist = ball1.radius + ball2.radius
    
    if dist < min_dist and dist > 0:
        # Normalize collision vector
        nx = dx / dist
        ny = dy / dist
        
        # Separate balls
        overlap = min_dist - dist
        ball1.x -= nx * overlap / 2
        ball1.y -= ny * overlap / 2
        ball2.x += nx * overlap / 2
        ball2.y += ny * overlap / 2
        
        # Relative velocity
        dvx = ball1.vx - ball2.vx
        dvy = ball1.vy - ball2.vy
        
        # Relative velocity along collision normal
        dvn = dvx * nx + dvy * ny
        
        if dvn > 0:  # Only if balls are approaching
            # Elastic collision with restitution
            j = -(1 + RESTITUTION) * dvn / (1 / ball1.mass + 1 / ball2.mass)
            
            # Apply impulse
            ball1.vx += j * nx / ball1.mass
            ball1.vy += j * ny / ball1.mass
            ball2.vx -= j * nx / ball2.mass
            ball2.vy -= j * ny / ball2.mass
            
            # Tangent for spin transfer
            tx = -ny
            ty = nx
            dvt = dvx * tx + dvy * ty
            
            # Transfer some angular momentum
            spin_transfer = 0.15
            ball1.angular_velocity -= dvt * spin_transfer / ball1.radius
            ball2.angular_velocity += dvt * spin_transfer / ball2.radius


def keep_balls_inside(ball: Ball, heptagon: Heptagon):
    """Ensure ball stays inside heptagon (backup containment)."""
    # Check if ball center is too far from heptagon center
    dx = ball.x - heptagon.cx
    dy = ball.y - heptagon.cy
    dist_from_center = math.hypot(dx, dy)
    
    # Maximum allowed distance (approximate inner radius minus ball radius)
    inner_radius = heptagon.radius * math.cos(math.pi / heptagon.sides)
    max_dist = inner_radius - ball.radius - 5
    
    if dist_from_center > max_dist and dist_from_center > 0:
        # Push ball back toward center
        factor = max_dist / dist_from_center
        ball.x = heptagon.cx + dx * factor
        ball.y = heptagon.cy + dy * factor
        
        # Reflect velocity
        nx = dx / dist_from_center
        ny = dy / dist_from_center
        vn = ball.vx * nx + ball.vy * ny
        if vn > 0:
            ball.vx -= 2 * vn * nx * RESTITUTION
            ball.vy -= 2 * vn * ny * RESTITUTION


class Simulation:
    """Main simulation class."""
    
    def __init__(self, root: tk.Tk):
        self.root = root
        self.root.title("Bouncing Balls in Spinning Heptagon")
        
        self.canvas = tk.Canvas(root, width=WIDTH, height=HEIGHT, bg="#1a1a2e")
        self.canvas.pack()
        
        # Create heptagon
        self.heptagon = Heptagon(CENTER_X, CENTER_Y, HEPTAGON_RADIUS)
        
        # Create balls
        self.balls: List[Ball] = []
        for i in range(NUM_BALLS):
            # Start from center with slight random offset to avoid perfect stack
            offset_x = (i % 5 - 2) * 3
            offset_y = (i // 5 - 2) * 3
            ball = Ball(
                x=CENTER_X + offset_x,
                y=CENTER_Y + offset_y,
                vx=np.random.uniform(-50, 50),
                vy=np.random.uniform(-20, 20),
                color=COLORS[i],
                number=i + 1,
                angular_velocity=np.random.uniform(-2, 2)
            )
            self.balls.append(ball)
        
        # Graphics handles
        self.heptagon_id = None
        self.ball_ids: List[int] = []
        self.text_ids: List[int] = []
        
        self.running = True
        self.last_time = None
        
        # Bind escape to quit
        self.root.bind('<Escape>', lambda e: self.quit())
        self.root.protocol("WM_DELETE_WINDOW", self.quit)
        
        self.animate()
    
    def quit(self):
        self.running = False
        self.root.destroy()
    
    def update_physics(self, dt: float):
        """Update all physics for one time step."""
        # Update heptagon rotation
        self.heptagon.update(dt)
        
        # Update balls
        for ball in self.balls:
            ball.update(dt)
        
        # Ball-wall collisions (multiple iterations for stability)
        for _ in range(3):
            for ball in self.balls:
                collide_ball_wall(ball, self.heptagon)
                keep_balls_inside(ball, self.heptagon)
        
        # Ball-ball collisions
        for i in range(len(self.balls)):
            for j in range(i + 1, len(self.balls)):
                collide_balls(self.balls[i], self.balls[j])
    
    def draw(self):
        """Draw the current state."""
        self.canvas.delete("all")
        
        # Draw heptagon
        vertices = self.heptagon.get_vertices()
        flat_coords = []
        for x, y in vertices:
            flat_coords.extend([x, y])
        
        self.canvas.create_polygon(
            flat_coords,
            outline="#e94560",
            fill="",
            width=4
        )
        
        # Draw subtle grid pattern on heptagon edges
        for i, ((x1, y1), (x2, y2)) in enumerate(self.heptagon.get_edges()):
            self.canvas.create_line(x1, y1, x2, y2, fill="#e94560", width=3)
        
        # Draw center marker
        self.canvas.create_oval(
            CENTER_X - 3, CENTER_Y - 3,
            CENTER_X + 3, CENTER_Y + 3,
            fill="#e94560", outline=""
        )
        
        # Draw balls (sorted by y for simple depth effect)
        sorted_balls = sorted(self.balls, key=lambda b: b.y)
        
        for ball in sorted_balls:
            # Draw ball shadow
            shadow_offset = 4
            self.canvas.create_oval(
                ball.x - ball.radius + shadow_offset,
                ball.y - ball.radius + shadow_offset,
                ball.x + ball.radius + shadow_offset,
                ball.y + ball.radius + shadow_offset,
                fill="#0a0a15",
                outline=""
            )
            
            # Draw ball
            self.canvas.create_oval(
                ball.x - ball.radius,
                ball.y - ball.radius,
                ball.x + ball.radius,
                ball.y + ball.radius,
                fill=ball.color,
                outline=self.darken_color(ball.color),
                width=2
            )
            
            # Draw highlight
            highlight_r = ball.radius * 0.4
            highlight_x = ball.x - ball.radius * 0.3
            highlight_y = ball.y - ball.radius * 0.3
            self.canvas.create_oval(
                highlight_x - highlight_r * 0.5,
                highlight_y - highlight_r * 0.5,
                highlight_x + highlight_r * 0.5,
                highlight_y + highlight_r * 0.5,
                fill=self.lighten_color(ball.color),
                outline=""
            )
            
            # Draw number with rotation
            # Calculate rotated position for number
            num_text = str(ball.number)
            
            # Use custom rotated text rendering
            self.draw_rotated_number(ball.x, ball.y, num_text, ball.angle, ball.radius)
    
    def draw_rotated_number(self, cx: float, cy: float, text: str, angle: float, radius: float):
        """Draw a number rotated around its center."""
        # Calculate font size based on ball radius
        font_size = int(radius * 0.9)
        
        # For visual rotation effect, we'll draw the number with offset based on angle
        # Since tkinter doesn't support rotated text easily, we show rotation via position wobble
        # and use the angle to show spin direction with a small indicator
        
        # Draw the number
        self.canvas.create_text(
            cx, cy,
            text=text,
            font=("Arial Black", font_size, "bold"),
            fill="#1a1a2e",
            anchor="center"
        )
        
        # Draw a small rotation indicator (dot that orbits the number)
        indicator_dist = radius * 0.6
        ind_x = cx + indicator_dist * math.cos(angle)
        ind_y = cy + indicator_dist * math.sin(angle)
        ind_r = 3
        self.canvas.create_oval(
            ind_x - ind_r, ind_y - ind_r,
            ind_x + ind_r, ind_y + ind_r,
            fill="#1a1a2e",
            outline=""
        )
    
    def darken_color(self, hex_color: str) -> str:
        """Darken a hex color."""
        r = int(hex_color[1:3], 16)
        g = int(hex_color[3:5], 16)
        b = int(hex_color[5:7], 16)
        factor = 0.7
        r = int(r * factor)
        g = int(g * factor)
        b = int(b * factor)
        return f"#{r:02x}{g:02x}{b:02x}"
    
    def lighten_color(self, hex_color: str) -> str:
        """Lighten a hex color."""
        r = int(hex_color[1:3], 16)
        g = int(hex_color[3:5], 16)
        b = int(hex_color[5:7], 16)
        factor = 0.4
        r = int(r + (255 - r) * factor)
        g = int(g + (255 - g) * factor)
        b = int(b + (255 - b) * factor)
        return f"#{r:02x}{g:02x}{b:02x}"
    
    def animate(self):
        """Main animation loop."""
        if not self.running:
            return
        
        # Use fixed time step for consistent physics
        self.update_physics(DT)
        self.draw()
        
        # Schedule next frame (~60 FPS)
        self.root.after(16, self.animate)


def main():
    root = tk.Tk()
    root.resizable(False, False)
    
    # Center window on screen
    root.update_idletasks()
    x = (root.winfo_screenwidth() - WIDTH) // 2
    y = (root.winfo_screenheight() - HEIGHT) // 2
    root.geometry(f"{WIDTH}x{HEIGHT}+{x}+{y}")
    
    sim = Simulation(root)
    root.mainloop()


if __name__ == "__main__":
    main()

